//
// Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
//

#import <Foundation/Foundation.h>
#import <AmazonIVSBroadcast/IVSBase.h>

NS_ASSUME_NONNULL_BEGIN

IVS_EXPORT
/// A configuration object describing the desired format of the published audio track associated the `IVSStageStream`.
@interface IVSLocalStageStreamAudioConfiguration : NSObject

/// The max bitrate for the publishing audio stream.
/// By default this is `64,000`.
@property (nonatomic, readonly) NSInteger maxBitrate;

/// Sets the bitrate for the publishing audio stream. This value must be between 12k and 128k
/// If the provided bitrate falls outside this range, bitrate will not be set and the provided outError will be set.
/// @param bitrate the  bitrate for the publishing audio stream
/// @param outError On input, a pointer to an error object. If an error occurs, the pointer is an NSError object that describes the error. If you don’t want error information, pass in nil.
/// @return if the set operation is successful or not.
- (BOOL)setMaxBitrate:(NSInteger)maxBitrate error:(NSError* __autoreleasing*)outError;

/// This property indicates whether the encoder is asked to encode in stereo or not.
/// By default this is `NO`.
///
/// @discussion
///
/// Depending on the bitrate or the subscriber's capabilities, not all subscribers may
/// hear stereo audio. There are many reasons a subscribe might not be able to play
/// stereo audio.
/// * The hardware they are using for audio output doesn't support stereo.
/// * Their decoder doesn't support decoding stereo audio.
/// * The audio mode they are using doesn't support stereo output.
///
/// On iOS, not all presets provided by `IVSStageAudioManager` can output stereo audio.
/// For best playback quality, IVS recommends using the `studio` or `subscribeOnly` presets.
/// Headsets with a dedicated left and right speaker also make the impact of stereo
/// more pronounced.
@property (nonatomic) BOOL stereo;

/// Setting this to `YES` will enable noise suppression.
///
/// By default this is `YES`.
@property (nonatomic) BOOL enableNoiseSuppression;

@end

NS_ASSUME_NONNULL_END
