//
// Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
//

#import <Foundation/Foundation.h>

/// Error codes for IVS Broadcast
typedef NS_ENUM(NSInteger, IVSBroadcastErrorCode) {
    // MARK: Net Errors

    /// Network connection failed during initialization due to system resource constraints. Verify network availability and retry the stream.
    IVSBroadcastErrorCodeNetSocketCreationFailed = 201,
    /// Network connection setup failed due to configuration issues. Check network connectivity and retry the stream.
    IVSBroadcastErrorCodeNetSocketConfigurationFailed = 202,
    /// Unable to establish network connection to the streaming server. Verify network connectivity and that the stream URL is accessible.
    IVSBroadcastErrorCodeNetSocketConnectFailed = 203,
    /// Failed to send data over the network connection. Check network stability and restart the stream.
    IVSBroadcastErrorCodeNetSocketSendFailed = 204,
    /// Failed to receive data from the streaming server. Check network stability and restart the stream.
    IVSBroadcastErrorCodeNetSocketRecvFailed = 205,
    /// The streaming server closed the connection unexpectedly. Verify no other users are streaming to this endpoint and restart the stream.
    IVSBroadcastErrorCodeNetSocketRecvFailedEOF = 206,
    /// Network connection closed unexpectedly during stream playback. Check network stability and restart the stream.
    IVSBroadcastErrorCodeNetSocketRemoteHangup = 207,
    /// Network error occurred during stream playback. Verify network connectivity and restart the stream.
    IVSBroadcastErrorCodeNetSocketRemoteError = 208,
    /// Network latency measurement is unavailable on this device. Streaming will continue without latency metrics.
    IVSBroadcastErrorCodeNetSocketGetRttFailedNotSupportedOnThisSystem = 212,
    /// Network send buffer configuration failed. Check network conditions and retry the stream.
    IVSBroadcastErrorCodeNetSocketSendLowWaterFailed = 213,
    /// Unable to resolve the streaming server hostname. Check network connectivity and verify the stream URL is correct.
    IVSBroadcastErrorCodeNetCouldNotResolveAddress = 214,
    /// Network metrics calculation failed due to invalid configuration. This is an internal error that should not affect streaming, but calculated transmission statistics may be impacted.
    IVSBroadcastErrorCodeNetGetAverageOutgoingRateFailed = 300,
    /// Network congestion measurement failed due to invalid configuration. This is an internal error that should not affect streaming, but calculated transmission statistics may be impacted.
    IVSBroadcastErrorCodeNetGetEstimatedCongestionLevelFailed = 301,
    /// Secure connection data transmission failed due to protocol violation. Restart the stream to re-establish the connection.
    IVSBroadcastErrorCodeNetTLSSendFailedBadWriteRetry = 400,
    /// Data transmission attempted before secure connection was established. Restart the stream.
    IVSBroadcastErrorCodeNetTLSSendFailedAttemptedToSendWhileHandshaking = 401,
    /// Data transmission attempted after connection was closed. Restart the stream to re-establish the connection.
    IVSBroadcastErrorCodeNetTLSSendFailedAttemptedToSendWhileDisconnected = 402,
    /// Data reception attempted before secure connection was established. Restart the stream.
    IVSBroadcastErrorCodeNetTLSRecvFailedAttemptedToRecvWhileHandshaking = 403,
    /// Data reception attempted after connection was closed. Restart the stream to re-establish the connection.
    IVSBroadcastErrorCodeNetTLSRecvFailedAttemptedToRecvWhileDisconnected = 404,
    /// The streaming server closed the connection. Verify no other users are streaming to this endpoint and restart the stream.
    IVSBroadcastErrorCodeNetTLSRecvFailedAttemptedToRecvAfterReceivingShutdownFromPeer = 405,
    /// Data reception attempted while connection is closing. This is expected during stream shutdown.
    IVSBroadcastErrorCodeNetTLSRecvFailedAttemptedToRecvWhileShuttingDown = 406,
    /// Secure connection initialization failed. This is usually an allocation failure, the device may not be able to allocate more memory to the process.
    IVSBroadcastErrorCodeNetTLSHandshakeFailedCouldNotCreateSslInstance = 407,
    /// Failed to send data over secure connection. Check network stability and restart the stream. This could also be caused by a firewall restricting access to the provided domain.
    IVSBroadcastErrorCodeNetTLSSendFailedOpenSSLError = 408,
    /// Failed to receive data over secure connection. Check network stability and restart the stream.
    IVSBroadcastErrorCodeNetTLSRecvFailedOpenSSLError = 409,
    /// Secure connection handshake failed. Verify network connectivity and restart the stream. This could also be caused by a firewall restricting access to the provided domain.
    IVSBroadcastErrorCodeNetTLSHandshakeFailedOpenSSLError = 410,
    /// Certificate verification setup failed during secure connection. Restart the stream to retry the connection. This could also be caused by a firewall restricting access to the provided domain.
    IVSBroadcastErrorCodeNetTLSFailedToGetX509Param = 411,
    /// Server hostname configuration failed during secure connection setup. Verify the stream URL and restart the stream. This could also be caused by a firewall restricting access to the provided domain.
    IVSBroadcastErrorCodeNetTLSSetSNIFailed = 412,
    /// Certificate hostname verification setup failed. Verify the stream URL is correct and restart the stream. This could also be caused by a firewall restricting access to the provided domain.
    IVSBroadcastErrorCodeNetTLSSetExpectedHostFailed = 413,
    /// Internal timing mechanism failed. This may prevent efficient DNS resolution.
    IVSBroadcastErrorCodeNetEventLoopFailedToCreateTimer = 500,
    /// Server did not accept WebSocket connection request.
    IVSBroadcastErrorCodeNetWebSocketHandshakeInvalidUpgradeResponse = 600,
    /// WebSocket handshake validation failed.
    IVSBroadcastErrorCodeNetWebSocketHandshakeInvalidAcceptKey = 601,
    /// Server selected a WebSocket protocol not supported by the client. Verify the events URL is correct and restart the session.
    IVSBroadcastErrorCodeNetWebSocketHandshakeProtocolNegotiationFailed = 602,
    /// WebSocket connection took too long to establish. Check network connectivity and retry the connection.
    IVSBroadcastErrorCodeNetWebSocketHandshakeTimeout = 603,
    /// Server rejected WebSocket connection with unexpected status code. Verify the events URL supports WebSocket connections.
    IVSBroadcastErrorCodeNetWebSocketHandshakeInvalidStatusCode = 604,
    /// Server response headers do not meet WebSocket protocol requirements. Verify the events URL is correct and supports WebSocket.
    IVSBroadcastErrorCodeNetWebSocketHandshakeInvalidHeaders = 605,
    /// Received malformed data from the streaming server. Restart the session to re-establish the connection.
    IVSBroadcastErrorCodeNetWebSocketFrameInvalidOpcode = 620,
    /// Received data with invalid length from the streaming server. Restart the connection to re-establish the connection.
    IVSBroadcastErrorCodeNetWebSocketFrameInvalidLength = 621,
    /// Server violated WebSocket protocol by sending masked data. Restart the session to re-establish the connection.
    IVSBroadcastErrorCodeNetWebSocketFrameInvalidMask = 622,
    /// Server sent control message exceeding protocol limit. Restart the session to re-establish the connection.
    IVSBroadcastErrorCodeNetWebSocketFrameControlFrameTooLarge = 623,
    /// Received text message with invalid character encoding from the server. Restart the session to re-establish the connection.
    IVSBroadcastErrorCodeNetWebSocketFrameInvalidUTF8 = 624,
    /// Server violated WebSocket protocol by setting reserved bits. Restart the session to re-establish the connection.
    IVSBroadcastErrorCodeNetWebSocketFrameReservedBitsSet = 625,
    /// Received partial data from the streaming server. This is typically handled automatically by waiting for more data and should not impact the session.
    IVSBroadcastErrorCodeNetWebSocketFrameIncompleteData = 626,
    /// Server violated WebSocket protocol by fragmenting a control message. Restart the session to re-establish the connection.
    IVSBroadcastErrorCodeNetWebSocketFrameControlFrameFragmented = 627,
    /// Received message exceeds maximum allowed size. Restart the session to re-establish the connection.
    IVSBroadcastErrorCodeNetWebSocketProtocolMessageTooLarge = 641,
    /// Attempted to send data before WebSocket connection was ready. This is an internal SDK and the session should be retried.
    IVSBroadcastErrorCodeNetWebSocketProtocolConnectionNotEstablished = 642,
    /// WebSocket connection took too long to establish. Check network connectivity and restart the session.
    IVSBroadcastErrorCodeNetWebSocketConnectionTimeout = 660,
    /// The streaming server closed the WebSocket connection. Check network connectivity and restart the session.
    IVSBroadcastErrorCodeNetWebSocketConnectionLost = 661,
    /// Server did not acknowledge connection close request in time. The connection will be terminated forcefully.
    IVSBroadcastErrorCodeNetWebSocketConnectionCloseTimeout = 662,
    /// The client sent a WebSocket PING on the idle WebSocket but no PONG was received within the timeout period. It is likely the socket has disconnected but hasn't yet reached the TCP idle timeout. The WebSocket will automatically attempt to reconnect.
    IVSBroadcastErrorCodeNetWebSocketConnectionPongTimeout = 663,

    // MARK: RTMP Errors

    /// The streaming server rejected the connection or encountered an error. Check your stream key and server configuration, then retry.
    IVSBroadcastErrorCodeRTMPNetConnectionErrorReceivedFromServer = 3007,
    /// Timed metadata payload exceeds the 10KB limit. Reduce the metadata size to stay within the allowed limit.
    IVSBroadcastErrorCodeRTMPNotStreamContentsTooLarge = 3110,
    /// Cannot start broadcast when already started. Stop the current broadcast and create a new BroadcastSession before starting a new stream.
    IVSBroadcastErrorCodeRTMPStateNotUnknown = 3200,
    /// Received data from server in an unexpected state. This indicates a protocol error; restart the broadcast session.
    IVSBroadcastErrorCodeRTMPUnexpectedStateProcessingData = 3201,
    /// The streaming server is using an incompatible RTMP version. Verify the server configuration and ensure it supports standard RTMP.
    IVSBroadcastErrorCodeRTMPUnsupportedRTMPVersion = 3202,
    /// Received invalid data from the streaming server. Verify the server URL is correct and supports RTMP streaming.
    IVSBroadcastErrorCodeRTMPDataDoesNotLookLikeRTMP = 3203,
    /// Server sent data before establishing the stream. This indicates a protocol error; restart the broadcast session.
    IVSBroadcastErrorCodeRTMPUninitializedChunkStream = 3204,
    /// Server sent control message on wrong stream. This indicates a protocol error; restart the broadcast session.
    IVSBroadcastErrorCodeRTMPControlMessageOnNonNetConnection = 3205,
    /// Server sent malformed chunk size message. This indicates a protocol error; restart the broadcast session.
    IVSBroadcastErrorCodeRTMPUnexpectedLengthSetChunkSize = 3206,
    /// Server requested invalid chunk size (must be 1-16777215 bytes). This indicates a protocol error; restart the broadcast session.
    IVSBroadcastErrorCodeRTMPUnexpectedChunkSizeValue = 3207,
    /// Server sent malformed abort message. This indicates a protocol error; restart the broadcast session.
    IVSBroadcastErrorCodeRTMPUnexpectedLengthAbortMessage = 3208,
    /// Server sent abort message with invalid stream ID. This indicates a protocol error; restart the broadcast session.
    IVSBroadcastErrorCodeRTMPUnexpectedChunkStreamIdAbort = 3209,
    /// Server tried to abort a stream that was never initialized. This indicates a protocol error; restart the broadcast session.
    IVSBroadcastErrorCodeRTMPAbortOnUninitializedChunkStream = 3210,
    /// Server sent malformed acknowledgement message. This indicates a protocol error; restart the broadcast session.
    IVSBroadcastErrorCodeRTMPUnexpectedLengthAckControl = 3211,
    /// Server sent malformed window acknowledgement size message. This indicates a protocol error; restart the broadcast session.
    IVSBroadcastErrorCodeRTMPUnexpectedLengthWindowAckSize = 3212,
    /// Server sent malformed bandwidth control message. This indicates a protocol error; restart the broadcast session.
    IVSBroadcastErrorCodeRTMPUnexpectedLengthSetPeerBandwidth = 3213,
    /// Server sent malformed user control message. This indicates a protocol error; restart the broadcast session.
    IVSBroadcastErrorCodeRTMPUnexpectedLengthUserControl = 3214,
    /// Server sent malformed ping request. This indicates a protocol error; restart the broadcast session.
    IVSBroadcastErrorCodeRTMPUnexpectedLengthPingRequest = 3215,
    /// Server sent message exceeding maximum allowed size. This indicates a protocol error; restart the broadcast session.
    IVSBroadcastErrorCodeRTMPAMFDecodeLimitExceeded = 3300,

    // MARK: Broadcast Errors

    /// Audio mixer received empty audio buffer. Ensure audio sources are providing valid data before mixing.
    IVSBroadcastErrorCodeBroadcastAudioMixerMissingData = 20004,
    /// Session must be configured before starting a broadcast. Call setup() before attempting to start streaming.
    IVSBroadcastErrorCodeBroadcastBroadcastSessionNotReady = 20100,
    /// Operation requires an active broadcast connection. Start streaming before calling this method.
    IVSBroadcastErrorCodeBroadcastBroadcastSessionNotStreaming = 20101,
    /// Cannot start a new broadcast while already streaming or running a network test. Stop the current session first.
    IVSBroadcastErrorCodeBroadcastBroadcastSessionAlreadyStreaming = 20102,
    /// Metadata payload exceeds the 10KB limit. Reduce the metadata size before sending.
    IVSBroadcastErrorCodeBroadcastBroadcastSessionMetadataTooLarge = 20103,
    /// Streaming URL or key exceeds the 64KB limit. Verify the endpoint and stream key are valid.
    IVSBroadcastErrorCodeBroadcastBroadcastSessionEndpointOrKeyTooLong = 20104,
    /// Streaming URL uses an unsupported protocol. Use rtmp:// or rtmps:// schemes.
    IVSBroadcastErrorCodeBroadcastBroadcastSessionInvalidScheme = 20105,
    /// Cannot add a slot with a duplicate name. Use a unique name or remove the existing slot first.
    IVSBroadcastErrorCodeBroadcastMixerAddSlotNameAlreadyExists = 20230,
    /// Cannot remove a slot that doesn't exist. Verify the slot name is correct.
    IVSBroadcastErrorCodeBroadcastMixerRemoveSlotNotFound = 20231,
    /// Video sample was ignored because no slot is bound to receive it. This is informational and typically expected behavior.
    IVSBroadcastErrorCodeBroadcastMixerSampleIgnoredNoSlot = 20232,
    /// Network bandwidth is insufficient for streaming. A stream can still be started, but it is expected to perform very poorly even at a low minimum bitrate.
    IVSBroadcastErrorCodeBroadcastConnectionTestSessionBandwidthTooLow = 20301,
    /// Network test was canceled before completion. Start a new test if you need recommended settings.
    IVSBroadcastErrorCodeBroadcastConnectionTestSessionCanceled = 20302,
    /// Video keyframe is missing required codec configuration data. Possible bad encoder output, restart the BroadcastSession to recreate the encoder.
    IVSBroadcastErrorCodeBroadcastBadSyncSample = 20400,
    /// Network latency exceeded acceptable limits and the broadcast was stopped. Improve network stability or reduce bitrate.
    IVSBroadcastErrorCodeBroadcastLatencyThresholdReached = 20401,
    /// Stream stop operation did not complete successfully. Wait 30 seconds before streaming to the same channel again.
    IVSBroadcastErrorCodeBroadcastStopFailedToComplete = 20402,
    /// Video pixel format is not supported by the device. Use a supported format like NV12 or RGBA.
    IVSBroadcastErrorCodeBroadcastRenderContextUnsupportedPixelFormat = 21000,
    /// Background image source received no valid images before finalization. Ensure images are submitted before stopping the source.
    IVSBroadcastErrorCodeBroadcastBackgroundVideoNoValidImagesReceived = 21200,
    /// Image buffer stride value is invalid for the specified dimensions. Verify stride matches pixel format requirements.
    IVSBroadcastErrorCodeBroadcastImageBufferBadStride = 21250,
    /// Image buffer size is too small for the specified dimensions and format. Check that the input buffer provided is valid.
    IVSBroadcastErrorCodeBroadcastImageBufferNotEnoughData = 21251,
    /// Image buffer size exceeds expected size for the specified dimensions and format. Verify buffer size calculation.
    IVSBroadcastErrorCodeBroadcastImageBufferTooMuchData = 21252,
    /// Image pixel format is not supported on this device. Use a supported format like NV12 or RGBA.
    IVSBroadcastErrorCodeBroadcastImageBufferUnsupportedFormat = 21253,
    /// Image buffer content size did not equal width * height * bits per pixel. Check to make sure the image buffer was not corrupted.
    IVSBroadcastErrorCodeBroadcastImageBufferContentSizeMismatch = 21254,
    /// Image buffer pixel format cannot be changed after creation. Create a new buffer with the desired format.
    IVSBroadcastErrorCodeBroadcastImageBufferFormatMismatch = 21255,
    /// Image buffer is invalid or was destroyed. Verify the buffer is properly initialized before use.
    IVSBroadcastErrorCodeBroadcastImageBufferNotValid = 21256,
    /// Image data pointer is null or invalid. Provide valid image data before submitting.
    IVSBroadcastErrorCodeBroadcastImageBufferBadData = 21257,
    /// Operation is not supported for this image buffer type. Use a compatible buffer or operation.
    IVSBroadcastErrorCodeBroadcastImageBufferNotSupported = 21258,
    /// Codec support configuration is not yet available. Retry codec discovery after a brief delay.
    IVSBroadcastErrorCodeBroadcastCodecDiscoveryDeviceConfigKeyMissing = 21300,
    /// Codec is disabled for this streaming channel.
    IVSBroadcastErrorCodeBroadcastCodecDiscoveryDeviceConfigDisabledChannel = 21302,
    /// HEVC codec is not supported on this device with the current configuration. Use AVC instead.
    IVSBroadcastErrorCodeBroadcastCodecDiscoveryDeviceConfigDisabledDevice = 21303,
    /// Video encoder produced frames with non-increasing timestamps. This device may not support the selected codec reliably.
    IVSBroadcastErrorCodeBroadcastVideoEncoderValidatorDecodeTimeDecreasing = 21304,
    /// Video encoder is missing required codec configuration data (SPS/PPS/VPS). This device may not support the selected codec reliably.
    IVSBroadcastErrorCodeBroadcastVideoEncoderValidatorDecodeMissingOrIncorrectCodecData = 21305,
    /// Video encoder is too slow or stopped responding during validation. This device may not support the selected codec or quality settings.
    IVSBroadcastErrorCodeBroadcastVideoEncoderValidatorEncodeTimeout = 21306,
    /// Video encoder produced an unexpected number of keyframes during validation. This device may not support the selected codec reliably.
    IVSBroadcastErrorCodeBroadcastVideoEncoderValidatorIncorrectKeyframeCount = 21307,
    /// Failed to establish streaming connection to the server. Check network connectivity and verify the streaming URL is correct.
    IVSBroadcastErrorCodeBroadcastNetConnectionConnectFailed = 21500,
    /// Failed to start publishing the stream to the server. Verify the stream key is valid and the channel is not already streaming.
    IVSBroadcastErrorCodeBroadcastNetStreamPublishFailed = 21501,
    /// Streaming URL uses an unsupported protocol. Use rtmp:// or rtmps:// schemes.
    IVSBroadcastErrorCodeBroadcastInvalidScheme = 21503,
    /// Streaming URL is missing a valid port number and no default is available. Verify the URL format is correct.
    IVSBroadcastErrorCodeBroadcastCouldNotDeterminePort = 21504,
    /// Streaming URL contains an invalid port number. Use a port between 1 and 65535.
    IVSBroadcastErrorCodeBroadcastInvalidPort = 21505,
    /// Audio statistics are unavailable because no audio data has been processed. Ensure audio sources are active and producing data.
    IVSBroadcastErrorCodeBroadcastAudioStatsNoData = 21600,
    /// Cannot embed an empty message in the video stream. Provide valid message data before calling this method.
    IVSBroadcastErrorCodeBroadcastEmbedMessagePayloadEmpty = 21700,
    /// Message data exceeds the 1KB limit for embedding in video frames. Reduce the message size to 1KB or less.
    IVSBroadcastErrorCodeBroadcastEmbedMessagePayloadTooLarge = 21701,
    /// Message repeat count is outside the valid range of 0-30. Adjust the repeat count to be within this range.
    IVSBroadcastErrorCodeBroadcastEmbedMessageInvalidRepeatCount = 21702,
    /// Message embedding rate limit exceeded (10KB per second). Reduce message frequency or size to stay within the limit.
    IVSBroadcastErrorCodeBroadcastEmbedMessageRateLimitExceeded = 21703,
    /// Video rendering failed because no output buffer was provided. Ensure a valid destination buffer is set before rendering.
    IVSBroadcastErrorCodeBroadcastCoreImageRenderContextMissingDestinationBuffer = 30000,
    /// Video rendering shader is missing from the SDK bundle. Verify the SDK was packaged correctly and all resources are included.
    IVSBroadcastErrorCodeBroadcastMetalRenderContextKernelNotFound = 30105,
    /// Video rendering initialization failed on this device. This indicates a device compatibility issue with Metal graphics.
    IVSBroadcastErrorCodeBroadcastMetalRenderContextSamplerCreationFailed = 30106,
    /// Video encoder failed repeatedly and cannot recover. This device may not support the selected codec or quality settings reliably.
    IVSBroadcastErrorCodeBroadcastVideoEncoderRecreateFailedTooManyTimes = 32001,
    /// SSL certificate validation encountered an internal error. Check network connectivity and verify the streaming endpoint uses valid certificates.
    IVSBroadcastErrorCodeBroadcastCertValidatorInternalError = 32400,
    /// Failed to load trusted certificate authorities for SSL validation. This indicates a system configuration issue.
    IVSBroadcastErrorCodeBroadcastCertValidatorFailedToLoadTrustRoots = 32401,
    /// SSL certificate hostname validation failed. Verify the streaming URL matches the certificate and the endpoint is correct.
    IVSBroadcastErrorCodeBroadcastCertValidatorFailedToValidateHost = 32402,
    /// SSL certificate chain is empty or invalid. Verify the streaming endpoint provides valid certificates.
    IVSBroadcastErrorCodeBroadcastCertValidatorFailedToFindTarget = 32403,
    /// SSL certificate validation failed. Verify the streaming endpoint uses valid, non-expired certificates.
    IVSBroadcastErrorCodeBroadcastCertValidatorFailedToValidateCert = 32404,
    /// Audio encoding failed during compression. The encoder will attempt to reset and continue streaming.
    IVSBroadcastErrorCodeBroadcastATAudioEncoderFillBuffer = 40001,
    /// Failed to configure audio encoder with required settings. Verify audio configuration is supported on this device.
    IVSBroadcastErrorCodeBroadcastATAudioEncoderSetProperty = 40002,
    /// Failed to retrieve audio encoder configuration. The encoder may not support the requested audio format.
    IVSBroadcastErrorCodeBroadcastATAudioEncoderGetProperty = 40003,
    /// Failed to initialize audio encoder. Verify audio configuration matches device capabilities.
    IVSBroadcastErrorCodeBroadcastATAudioEncoderCreate = 40004,
    /// Video encoder repeatedly failed and cannot recover. Stop and restart the broadcast session.
    IVSBroadcastErrorCodeBroadcastVTVideoEncoderRecreateFailedTooManyTimes = 40100,
    /// Video frame encoding failed. The encoder has stopped and the broadcast session must be restarted.
    IVSBroadcastErrorCodeBroadcastVTVideoEncoderFailedToEncodeFrame = 40101,
    /// Failed to extract encoded video data from compression buffer. The broadcast session must be restarted.
    IVSBroadcastErrorCodeBroadcastVTVideoEncoderExtractBufferFromCompressedFrameError = 40102,
    /// Failed to configure video encoder with required settings. Verify video configuration is supported on this device.
    IVSBroadcastErrorCodeBroadcastVTVideoEncoderSetProperty = 40103,
    /// Failed to initialize video encoder. Verify video configuration matches device capabilities.
    IVSBroadcastErrorCodeBroadcastVTVideoEncoderCreate = 40104,
    /// Failed to query video encoder capabilities. The device may not support the requested video format.
    IVSBroadcastErrorCodeBroadcastVTVideoEncoderGetSupportedProperties = 40105,
    /// Video encoder failed to initialize for encoding. Stop and restart the broadcast session.
    IVSBroadcastErrorCodeBroadcastVTVideoEncoderPrepareToEncodeFrames = 40106,
    /// Unable to identify the hardware encoder in use. This is informational and does not affect streaming.
    IVSBroadcastErrorCodeBroadcastVTVideoEncoderFailedToReadEncoderName = 40107,
    /// Low latency encoding mode is unavailable on this device, using standard encoder instead. Streaming will continue with slightly higher latency.
    IVSBroadcastErrorCodeBroadcastVTVideoEncoderFailedToUseLowLatency = 40108,
    /// Video encoder rejected the requested bitrate setting. The encoder will use a default bitrate instead.
    IVSBroadcastErrorCodeBroadcastVTVideoEncoderFailedToSetBitrate = 40109,
    /// Failed to allocate video buffer from the buffer pool. This device may have insufficient memory or resources.
    IVSBroadcastErrorCodeBroadcastImageBufferFailedToClearBuffer = 40200,
    /// Failed to lock video buffer for processing. Check the platform error code for device-specific details.
    IVSBroadcastErrorCodeBroadcastImageBufferFailedToLock = 40201,
    /// Failed to unlock video buffer after processing. Check the platform error code for device-specific details.
    IVSBroadcastErrorCodeBroadcastImageBufferFailedToUnlock = 40202,
    /// Failed to initialize video buffer pool for rendering. This device may have insufficient memory or resources.
    IVSBroadcastErrorCodeBroadcastCoreVideoRenderContextPoolCreateFailed = 40250,
    /// Metal rendering shaders are missing from the SDK bundle. Verify the SDK was packaged correctly and all resources are included.
    IVSBroadcastErrorCodeBroadcastMetalRenderContextLibraryNotLoaded = 40300,
    /// Failed to create Metal texture from video buffer. If using a custom image source, ensure kCVPixelBufferMetalCompatibilityKey is set to true when creating buffers.
    IVSBroadcastErrorCodeBroadcastMetalRenderContextCreateTextureFromImage = 40301,
    /// Video rendering initialization failed on this device. This indicates a device compatibility issue with Metal graphics.
    IVSBroadcastErrorCodeBroadcastMetalRenderContextRenderPipelineCreationFailed = 40302,
    /// Audio processing failed to extract audio data from the buffer. This indicates an audio format or buffer structure issue.
    IVSBroadcastErrorCodeBroadcastIVSAudioSourceExtractBufferFailed = 40400,

    // MARK: Real-Time Errors

    /// Stage token is invalid or malformed. Generate a new token and create a new stage session.
    IVSBroadcastErrorCodeRealTimeStageInvalidToken = 1000,
    /// Stage token has expired. Generate a new token and create a new stage session.
    IVSBroadcastErrorCodeRealTimeStageExpiredToken = 1001,
    /// Cannot perform operation because no active stage session exists. Join a stage before attempting this operation.
    IVSBroadcastErrorCodeRealTimeNoSession = 1002,
    /// Cannot join stage because a join operation is already in progress or completed. Wait for the current operation to finish.
    IVSBroadcastErrorCodeRealTimeAlreadyJoiningStage = 1005,
    /// The Stage session is not ready due to a failure is the setup process. Check for errors during initialization, and recreate the local Stage instance.
    IVSBroadcastErrorCodeRealTimeStageNotReady = 1011,
    /// Cannot subscribe to your own participant stream. Subscribe to other participants instead. This should not be possible using the StageStrategy.
    IVSBroadcastErrorCodeRealTimeSelfSubscribeNotSupport = 1012,
    /// Participant not found in the stage. This is likely because the remote participant unpublished at the same time this SDK attempted to subscribe.
    IVSBroadcastErrorCodeRealTimeNoParticipantToSubscribeTo = 1013,
    /// Failed to publish audio/video to the stage after multiple retry attempts. Check network connectivity and retry.
    IVSBroadcastErrorCodeRealTimePublishFailed = 1015,
    /// Failed to subscribe to participant stream after multiple retry attempts. Check network connectivity and retry.
    IVSBroadcastErrorCodeRealTimeSubscribeFailed = 1016,
    /// Failed to create WebRTC offer for publishing. Retry the publish operation.
    IVSBroadcastErrorCodeRealTimeOfferCreationFailPublish = 1020,
    /// Failed to create WebRTC offer for subscribing. Retry the subscribe operation.
    IVSBroadcastErrorCodeRealTimeOfferCreationFailSubscribe = 1021,
    /// Publish or subscribe operation was cancelled. This is expected when leaving or unsubscribing in the middle of an operation.
    IVSBroadcastErrorCodeRealTimePubSubAborted = 1023,
    /// Stage has reached maximum participant capacity. Wait for a participant to leave or create a new stage.
    IVSBroadcastErrorCodeRealTimeStageAtCapacity = 1024,
    /// An unexpected server error occurred. Retry the operation or contact support if the issue persists.
    IVSBroadcastErrorCodeRealTimeInternalServerError = 1025,
    /// Stage token lacks required permissions or has been revoked. Generate a new token with appropriate permissions.
    IVSBroadcastErrorCodeRealTimeStageTokenRejected = 1026,
    /// Stage authentication failed during join operation. The stage may have been deleted or is no longer available.
    IVSBroadcastErrorCodeRealTimeStageAuthenticationError = 1027,
    /// Input video resolution does not meet stage requirements. Increase camera resolution or adjust stage configuration.
    IVSBroadcastErrorCodeRealTimeInsufficientInputResolution = 1050,
    /// The token provided to the exchange token API was invalid and couldn't be parsed. Create a new token and try again.
    IVSBroadcastErrorCodeRealTimeTokenExchangeInvalidToken = 1051,
    /// The token provided to the exchange token API was not compatible with the existing Stage. This is likely because properties that must remain constant (jti, topic, etc) had changed.
    IVSBroadcastErrorCodeRealTimeTokenExchangeIncompatibleToken = 1052,
    /// Internal stage event system is not initialized. Restart the stage session.
    IVSBroadcastErrorCodeRealTimeMultihostGroupStatePipelineMissingMultihostEventBus = 1100,
    /// Internal event generation system is not initialized. Restart the stage session.
    IVSBroadcastErrorCodeRealTimeMultihostGroupStatePipelineMissingEventGenerator = 1101,
    /// Stage or participant resource no longer exists on the server. The stage may have been deleted or the session expired.
    IVSBroadcastErrorCodeRealTimeResourceDeleted = 1206,
    /// Disconnected from stage by the server for an unspecified reason. Rejoin the stage with a new token if needed.
    IVSBroadcastErrorCodeRealTimeRemoteDisconnectReasonUnknown = 1208,
    /// Disconnected because the stage was deleted.
    IVSBroadcastErrorCodeRealTimeRemoteDisconnectStageDeleted = 1209,
    /// Disconnected because another participant joined with the same token. Use unique tokens for each participant.
    IVSBroadcastErrorCodeRealTimeRemoteDisconnectTokenReused = 1210,
    /// Disconnected from stage by server request. Rejoin the stage with a new token if needed.
    IVSBroadcastErrorCodeRealTimeRemoteDisconnectParticipantDisconnected = 1211,
    /// Invalid response from the server for a token exchange request. Reconnect to the Stage using the new token, or create a new token and connect again.
    IVSBroadcastErrorCodeRealTimeTokenExchangeInvalidResponse = 1230,
    /// The token exchange request was rejected. The error comment will contain more details. A new token must be created and used to join a new SDK Stage instance.
    IVSBroadcastErrorCodeRealTimeTokenExchangeFailedResponse = 1231,
    /// The token exchange request timed out. Because a token exchange puts the SDK into an in between state of 2 tokens while waiting for confirmation from the backend, a timeout is enforced to prevent undefined behaviors.
    IVSBroadcastErrorCodeRealTimeTokenExchangeTimeout = 1232,
    /// Failed to connect to stage after multiple retry attempts. Check network connectivity and retry joining the stage.
    IVSBroadcastErrorCodeRealTimeEventsRetriesExhausted = 1300,
    /// Media connection lost due to network error. Check network connectivity and rejoin the stage.
    IVSBroadcastErrorCodeRealTimePeerConnectionNetworkError = 1400,
    /// Failed to establish media connection with participant. Retry the publish or subscribe operation.
    IVSBroadcastErrorCodeRealTimePeerConnectionSetRemoteDescriptionFail = 1401,
    /// Received invalid connection data from participant.
    IVSBroadcastErrorCodeRealTimePeerConnectionParseRemoteDescriptionError = 1404,
    /// Participant disconnected unexpectedly. The participant may have left the stage or lost network connectivity.
    IVSBroadcastErrorCodeRealTimePeerConnectionUnexpectedRemoteClose = 1407,
    /// Failed to configure video settings for the connection. Check system resources, this may have been a memory allocation failure.
    IVSBroadcastErrorCodeRealTimePeerConnectionSetParametersError = 1410,
    /// Cannot request keyframe because video sender is not ready. Ensure video is being published.
    IVSBroadcastErrorCodeRealTimePeerConnectionRequestKeyframeNoSender = 1411,
    /// Failed to configure media transceivers for the connection. This is a libWebRTC failure and may indicate the system is low on resources.
    IVSBroadcastErrorCodeRealTimePeerConnectionAddTransceiversFailed = 1412,
    /// Video sender is not ready for configuration. Ensure video is being published before adjusting settings.
    IVSBroadcastErrorCodeRealTimePeerConnectionVideoSenderNotReady = 1413,
    /// Video encoding parameters are not ready. Wait for the connection to fully establish before adjusting video settings.
    IVSBroadcastErrorCodeRealTimePeerConnectionVideoRtpParametersNotReady = 1414,
    /// Failed to update video encoding settings. This is a libWebRTC failure and may indicate the system is low on resources, or the provided configuration is not supported.
    IVSBroadcastErrorCodeRealTimePeerConnectionSetVideoRtpParametersFailed = 1415,
    /// Connection operation failed due to invalid configuration. The error message will contain more detailed information.
    IVSBroadcastErrorCodeRealTimePeerConnectionMethodNotSupportedError = 1420,
    /// Failed to initialize media connection. This is a libWebRTC failure and may indicate the system is low on resources. Restat the session to try again.
    IVSBroadcastErrorCodeRealTimePeerConnectionInitFail = 1421,
    /// Operation failed because the requested media type is not available. Ensure audio or video is being published or subscribed before requesting RTC stats.
    IVSBroadcastErrorCodeRealTimePeerConnectionNoMatchingMediaType = 1423,
    /// Media connection is taking too long to establish. This operation will be automatically retried, but this may indicate a network connectivity issue.
    IVSBroadcastErrorCodeRealTimePeerConnectionTimedOutConnecting = 1424,
    /// WebRTC connection operation failed. This operation will be automatically retried, but this may indicate a network connectivity issue.
    IVSBroadcastErrorCodeRealTimePeerConnectionOnFailure = 1425,
    /// Media connection failed to establish network path. This operation will be automatically retried, but this may indicate a network connectivity issue.
    IVSBroadcastErrorCodeRealTimePeerConnectionIceConnectionFailed = 1426,
    /// Cannot retrieve video sender configuration because sender is not initialized. Ensure video is being published.
    IVSBroadcastErrorCodeRealTimePeerConnectionCallbackSenderNotReady = 1427,
    /// Cannot retrieve video receiver configuration because receiver is not initialized. Ensure video is being subscribed.
    IVSBroadcastErrorCodeRealTimePeerConnectionCallbackReceiverNotReady = 1428,
    /// Cannot perform operation because media connection is not available. Ensure publish or subscribe is active.
    IVSBroadcastErrorCodeRealTimeNoValidPeerConnection = 1430,
    /// No audio packets are being sent through the publish connection. Check that audio input is active and not muted.
    IVSBroadcastErrorCodeRealTimeNoPublishAudioPacketsSent = 1450,
    /// No video packets are being sent through the publish connection. Check that video samples are being submitted.
    IVSBroadcastErrorCodeRealTimeNoPublishVideoPacketsSent = 1451,
    /// No audio packets are being received from the subscribed participant. The participant may have stopped publishing audio or network connectivity is lost.
    IVSBroadcastErrorCodeRealTimeNoSubscribeAudioPacketsReceived = 1452,
    /// No video packets are being received from the subscribed participant. The participant may have stopped publishing video without muting or network connectivity is lost.
    IVSBroadcastErrorCodeRealTimeNoSubscribeVideoPacketsReceived = 1453,
    /// Failed to set video quality layer for participant stream. This operation will be automatically retried, but this may indicate a network connectivity issue.
    IVSBroadcastErrorCodeRealTimeSetLayerRequestFailed = 1510,
    /// Video quality layer selection is being rate limited. This operation will be automatically retried after a delay.
    IVSBroadcastErrorCodeRealTimeSetLayerRequestThrottled = 1511,
    /// Failed to retrieve available video quality layers for participant stream. This operation will be automatically retried, but this may indicate a network connectivity issue.
    IVSBroadcastErrorCodeRealTimeGetLayerRequestFailed = 1512,
    /// Video quality layer query is being rate limited. This operation will be automatically retried after a delay.
    IVSBroadcastErrorCodeRealTimeGetLayerRequestThrottled = 1513,

};
