//
// Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
//

#import <Foundation/Foundation.h>
#import <AmazonIVSBroadcast/IVSBase.h>

NS_ASSUME_NONNULL_BEGIN

@class IVSCustomJitterBufferDelay;

/// A jitter buffer delay configuration. It can either be a preset or explicit value
IVS_EXPORT
@interface IVSJitterBufferDelay : NSObject

/// A preset for a minimum possible delay target.
+ (instancetype) default;
/// A preset for a minimum delay target that is high enough to deal with small network glitch.
+ (instancetype) low;
/// A preset for a minimum delay target that is high enough to deal with network glitch
+ (instancetype) medium;
/// A preset for a minimum delay target that is high enough to deal with relatively large network glitch.
+ (instancetype) high;
/// An explicit jitter buffer delay value
+ (IVSCustomJitterBufferDelay *) customWithTimeInterval:(NSTimeInterval)interval;

@end

/// A custom jitter buffer delay configuration.
IVS_EXPORT
@interface IVSCustomJitterBufferDelay : IVSJitterBufferDelay

///
/// The delay of the current jitter buffer configuration.
@property (nonatomic, readonly) NSTimeInterval interval;

@end

/// A configuration object describing jitter buffer configuration
IVS_EXPORT
@interface IVSJitterBufferConfiguration : NSObject

/// The current jitter buffer minimum delay.
///
/// By default this is `[IVSJitterBufferDelay default]`.
@property (strong, nonatomic, nonnull, readonly) IVSJitterBufferDelay* minDelay;

/// Sets the jitter buffer minimum delay for subscribing. This value must be a preset or explicit value between 0 and 10.
/// If the provided delay falls outside this range, the delay will not be set and the provided outError will be set.
/// @param minDelay The jitter buffer minimum delay for subscribing
/// @param outError On input, a pointer to an error object. If an error occurs, the pointer is an NSError object that describes the error. If you don’t want error information, pass in nil.
/// @return if the set operation is successful or not.
- (BOOL)setMinDelay:(IVSJitterBufferDelay*)minDelay error:(NSError* __autoreleasing*)outError;

@end

NS_ASSUME_NONNULL_END
