//
// Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
//

#import <Foundation/Foundation.h>
#import <AmazonIVSBroadcast/IVSBase.h>
#import <AmazonIVSBroadcast/IVSStageStream.h>

@class IVSRemoteStageStreamLayer;
@class IVSRemoteStageStreamLayerConstraints;

NS_ASSUME_NONNULL_BEGIN

@class IVSRemoteStageStream;

/// The reasons for a layer changing.
typedef NS_ENUM(NSInteger, IVSRemoteStageStreamLayerSelectedReason) {
    /// When a layer is unavailable, this means the SDK or SFU can no longer
    /// surface the layer, and a new best effort layer has been selected.
    /// Layer availability will also be notified via the didChangeLayers event.
    IVSRemoteStageStreamLayerSelectedReasonUnavailable = 0,
    /// The preferred layer set by the developer has been selected.
    IVSRemoteStageStreamLayerSelectedReasonSelected,
} NS_SWIFT_NAME(IVSRemoteStageStream.LayerSelectedReason);

IVS_EXPORT
/// A delegate that provides information about the associated `IVSRemoteStageStream`.
@protocol IVSRemoteStageStreamDelegate <IVSStageStreamDelegate>

@optional
/// The layers of associated remote stream has changed.
/// @param stream The associated remote stream.
/// @param layers The layers.
- (void)stream:(IVSRemoteStageStream *)stream didChangeLayers:(NSArray<IVSRemoteStageStreamLayer *> *)layers;

/// The selected layer has changed.
/// @param stream The associated remote stream.
/// @param layer The selected layer.
/// @param reason The reason for a layer changing.
- (void)stream:(IVSRemoteStageStream *)stream didSelectLayer:(nullable IVSRemoteStageStreamLayer *)layer reason:(IVSRemoteStageStreamLayerSelectedReason)reason;

/// The adaption has changed.
/// @param stream The associated remote stream.
/// @param adaption Whether dynamic adaption is enabled.
- (void)stream:(IVSRemoteStageStream *)stream didChangeAdaption:(BOOL)adaption;

@end

IVS_EXPORT
/// A remote stage stream that is used to subscribe.
@interface IVSRemoteStageStream : IVSStageStream

IVS_INIT_UNAVAILABLE

/// A delegate that can provide updates about this stream.
@property (nonatomic, weak) id<IVSRemoteStageStreamDelegate> delegate;

/// The layers for the remote stream.
///
/// @note this must be called on the main thread
@property (nonatomic, strong, readonly) NSArray<IVSRemoteStageStreamLayer *> *layers;

/// The selected layer.
///
/// @note this must be called on the main thread
@property (nullable, nonatomic, strong, readonly) IVSRemoteStageStreamLayer *selectedLayer;

/// The layer with the largest combined surface area (width x height).
///
/// @note this must be called on the main thread
@property (nonatomic, strong, readonly) IVSRemoteStageStreamLayer *highestQualityLayer;

/// The layer with the smalles combined surface area (width x height).
///
/// @note this must be called on the main thread
@property (nonatomic, strong, readonly) IVSRemoteStageStreamLayer *lowestQualityLayer;

/// This represents if dynamic simulcast adaption is enabled.
///
/// @note this must be called on the main thread
@property (nonatomic, readonly) BOOL isAdapting;

/// The list of available layers which match the constraints given.
/// @param constraints The layer constraints.
///
/// @note this must be called on the main thread
- (NSArray<IVSRemoteStageStreamLayer *> *)layersWithConstraints:(IVSRemoteStageStreamLayerConstraints *)constraints;

/// The first layer which match the constraints given.
/// @param constraints The layer constraints.
/// @param defaultValue A fallback value when there is no matched layer.
///
/// @note this must be called on the main thread
- (IVSRemoteStageStreamLayer *)firstLayerWithConstraints:(IVSRemoteStageStreamLayerConstraints *)constraints or:(IVSRemoteStageStreamLayer *)defaultValue;

@end

NS_ASSUME_NONNULL_END
